/*
 * sstar_rpmsg.c- Sigmastar
 *
 * Copyright (c) [2019~2020] SigmaStar Technology.
 *
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License version 2 for more details.
 *
 */
/*
 * sstar_rpmsg.c
 */

#include <linux/clk.h>
#include <linux/err.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/notifier.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_device.h>
#include <linux/of_irq.h>
#include <linux/platform_device.h>
#include <linux/rpmsg.h>
#include <linux/slab.h>
#include <linux/virtio.h>
#include <linux/virtio_config.h>
#include <linux/virtio_ids.h>
#include <linux/virtio_ring.h>
#include <linux/arm-smccc.h>
#include <linux/smp.h>
#include <linux/delay.h>
#include <linux/version.h>
#include <linux/kthread.h>

#include "sstar_rpmsg.h"
#include "ms_platform.h"
#include "registers.h"
#include "drv_dualos.h"
#include "cam_inter_os.h"

#define TARGET_BITS_CORE0    (1 << 16)
#define TARGET_BITS_CORE1    (1 << 17)
#define NSATT_BITS_GROUP0    (0 << 15)
#define NSATT_BITS_GROUP1    (1 << 15)
#define SGIINTID_BITS_08     (8)
#define SGIINTID_BITS_09     (9)
#define SGIINTID_BITS_10     (10)
#define SGIINTID_BITS_11     (11)
#define SGIINTID_BITS_15     (15)

typedef struct {
    unsigned int        arg0_l;
    unsigned int        arg0_h;
    unsigned int        arg1_l;
    unsigned int        arg1_h;
    unsigned int        arg2_l;
    unsigned int        arg2_h;
    unsigned int        arg3_l;
    unsigned int        arg3_h;
    unsigned int        ret_l;
    unsigned int        ret_h;
} interos_call_mbox_args_t;

struct sstar_virdev {
	struct virtio_device vdev;
	unsigned int vring[2];
	struct virtqueue *vq[2];
	int base_vq_id;
	int num_of_vqs;
	struct notifier_block nb;

	wait_queue_head_t wq;
	struct task_struct *task;
};

struct sstar_rpmsg_vproc {
	char *rproc_name;
	struct mutex lock;
	int vdev_nums;
#define MAX_VDEV_NUMS	1
	struct sstar_virdev ivdev[MAX_VDEV_NUMS];
};

struct sstar_rpmsg_mbox {
	const char *name;
	struct blocking_notifier_head notifier;
};

static struct sstar_rpmsg_mbox rpmsg_mbox = {
	.name	= "rtos",
};

static struct sstar_rpmsg_vproc sstar_rpmsg_vprocs[] = {
	{
		.rproc_name	= "rtos",
	},
};

static u32 in_idx, out_idx;

/*
 * For now, allocate 256 buffers of 512 bytes for each side. each buffer
 * will then have 16B for the msg header and 496B for the payload.
 * This will require a total space of 256KB for the buffers themselves, and
 * 3 pages for every vring (the size of the vring depends on the number of
 * buffers it supports).
 */
#define RPMSG_NUM_BUFS		(512)
#define RPMSG_BUF_SIZE		(512)
#define RPMSG_BUFS_SPACE	(RPMSG_NUM_BUFS * RPMSG_BUF_SIZE)

/*
 * The alignment between the consumer and producer parts of the vring.
 * Note: this is part of the "wire" protocol. If you change this, you need
 * to update your BIOS image as well
 */
#define RPMSG_VRING_ALIGN	(4096)

/* With 256 buffers, our vring will occupy 3 pages */
#define RPMSG_RING_SIZE	((DIV_ROUND_UP(vring_size(RPMSG_NUM_BUFS / 2, \
                          RPMSG_VRING_ALIGN), PAGE_SIZE)) * PAGE_SIZE)

#define to_sstar_virdev(vd) container_of(vd, struct sstar_virdev, vdev)
#define to_sstar_rpdev(vd, id) container_of(vd, struct sstar_rpmsg_vproc, ivdev[id])

struct sstar_rpmsg_vq_info {
	__u16 num;	/* number of entries in the virtio_ring */
	__u16 vq_id;	/* a globaly unique index of this virtqueue */
	void *addr;	/* address where we mapped the virtio ring */
	struct sstar_rpmsg_vproc *rpdev;
};

static unsigned long sstar_rpmsg_smc(u32 type)
{
	struct arm_smccc_res res;

	arm_smccc_smc(type, 0, 0, 0,
		0, 0, 0, 0, &res);
	return res.a0;
}

void sstar_rpmsg_reroute_smc(void)
{
	sstar_rpmsg_smc(INTEROS_SC_L2R_RPMSG_NOTIFY);
}

int sstar_rpmsg_isr(void)
{
	struct sstar_rpmsg_vproc *vproc = &sstar_rpmsg_vprocs[0];

	++in_idx;
	dsb();
	wake_up_all(&vproc->ivdev[0].wq);
	return 0;
}

static u64 sstar_rpmsg_get_features(struct virtio_device *vdev)
{
	/* VIRTIO_RPMSG_F_NS has been made private */
	return 1 << 0;
}

static int sstar_rpmsg_finalize_features(struct virtio_device *vdev)
{
	/* Give virtio_ring a chance to accept features */
	vring_transport_features(vdev);
	return 0;
}

#define GICD_SGIR	0x0F00
#define GICD_BASE	0xF4001000
#define GICD_WRITEL(a,v)    (*(volatile unsigned int *)(u32)(GICD_BASE + a) = (v))

#if defined(CONFIG_SMP)
static void sstar_rpmsg_send_SGI(int cpu, int no)
{
	GICD_WRITEL(GICD_SGIR, (1 << (cpu + 16)) | (1 << 15) | no);
}
#endif

/* kick the remote processor, and let it know which virtqueue to poke at */
static bool sstar_rpmsg_notify(struct virtqueue *vq)
{
#if defined(CONFIG_SMP)
	unsigned int rpmsg = 0;
	struct sstar_rpmsg_vq_info *rpvq = vq->priv;
	int cpu;

	rpmsg = rpvq->vq_id << 16;
	cpu = get_cpu();
	if (cpu == 0x0)
		sstar_rpmsg_smc(INTEROS_SC_L2R_RPMSG_NOTIFY);
	else
		sstar_rpmsg_send_SGI(0x0, IPI_NR_LINUX_2_RTOS_RPMSG);
	put_cpu();
#elif defined(CONFIG_SS_AMP)
	struct arm_smccc_res res;

	arm_smccc_smc(INTEROS_SC_L2R_CALL, TARGET_BITS_CORE1, NSATT_BITS_GROUP0, SGIINTID_BITS_15, 0, 0, 0, 0, &res);
#else
	sstar_rpmsg_smc(INTEROS_SC_L2R_RPMSG_NOTIFY);
#endif
	return true;
}

static int sstar_rpmsg_callback(struct notifier_block *this,
				unsigned long index, void *data)
{
	u32 id = 0;
	struct sstar_virdev *virdev;

	virdev = container_of(this, struct sstar_virdev, nb);

	/*
	 * We can't known which virtqueue triggers the interrupt,
	 * so let's iterate all the virtqueues.
	 */
	id = virdev->base_vq_id;
	for (; id < virdev->num_of_vqs; id++)
		vring_interrupt(id, virdev->vq[id]);

	return NOTIFY_DONE;
}

int sstar_rpmsg_register_nb(const char *name, struct notifier_block *nb)
{
	if ((name == NULL) || (nb == NULL))
		return -EINVAL;

	if (!strcmp(rpmsg_mbox.name, name))
		blocking_notifier_chain_register(&(rpmsg_mbox.notifier), nb);
	else
		return -ENOENT;

	return 0;
}

int sstar_rpmsg_unregister_nb(const char *name, struct notifier_block *nb)
{
	if ((name == NULL) || (nb == NULL))
		return -EINVAL;

	if (!strcmp(rpmsg_mbox.name, name))
		blocking_notifier_chain_unregister(&(rpmsg_mbox.notifier),
						nb);
	else
		return -ENOENT;

	return 0;
}

static struct virtqueue *rp_find_vq(struct virtio_device *vdev,
				unsigned int index,
				void (*callback)(struct virtqueue *vq),
				const char *name)
{
	struct sstar_virdev *virdev = to_sstar_virdev(vdev);
	struct sstar_rpmsg_vproc *rpdev = to_sstar_rpdev(virdev,
							virdev->base_vq_id / 2);
	struct sstar_rpmsg_vq_info *rpvq;
	struct virtqueue *vq;
	int err;

	rpvq = kmalloc(sizeof(*rpvq), GFP_KERNEL);
	if (!rpvq)
		return ERR_PTR(-ENOMEM);

	/* ioremap'ing normal memory, so we cast away sparse's complaints */
	rpvq->addr = (__force void *) ioremap_nocache(virdev->vring[index],
						RPMSG_RING_SIZE);
	if (!rpvq->addr) {
		err = -ENOMEM;
		goto free_rpvq;
	}

	memset(rpvq->addr, 0, RPMSG_RING_SIZE);

	pr_debug("vring%d: phys 0x%x, virt 0x%p\n", index, virdev->vring[index],
		rpvq->addr);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,12,0)
	vq = vring_new_virtqueue(index, RPMSG_NUM_BUFS / 2, RPMSG_VRING_ALIGN,
				vdev, true, false, rpvq->addr, sstar_rpmsg_notify, callback,
				name);
#else
	vq = vring_new_virtqueue(index, RPMSG_NUM_BUFS / 2, RPMSG_VRING_ALIGN,
				vdev, true, rpvq->addr, sstar_rpmsg_notify, callback,
				name);
#endif
	if (!vq) {
		pr_err("vring_new_virtqueue failed\n");
		err = -ENOMEM;
		goto unmap_vring;
	}

	virdev->vq[index] = vq;
	vq->priv = rpvq;
	/* system-wide unique id for this virtqueue */
	rpvq->vq_id = virdev->base_vq_id + index;
	rpvq->rpdev = rpdev;
	mutex_init(&rpdev->lock);

	return vq;

unmap_vring:
	/* iounmap normal memory, so make sparse happy */
	iounmap((__force void __iomem *) rpvq->addr);
free_rpvq:
	kfree(rpvq);
	return ERR_PTR(err);
}

static void sstar_rpmsg_del_vqs(struct virtio_device *vdev)
{
	struct virtqueue *vq, *n;
	struct sstar_virdev *virdev = to_sstar_virdev(vdev);
	struct sstar_rpmsg_vproc *rpdev = to_sstar_rpdev(virdev,
							virdev->base_vq_id / 2);

	list_for_each_entry_safe(vq, n, &vdev->vqs, list) {
		struct sstar_rpmsg_vq_info *rpvq = vq->priv;

		iounmap(rpvq->addr);
		vring_del_virtqueue(vq);
		kfree(rpvq);
	}

	if (&virdev->nb)
		sstar_rpmsg_unregister_nb((const char *)rpdev->rproc_name,
					&virdev->nb);
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,12,0)
static int sstar_rpmsg_find_vqs(struct virtio_device *vdev, unsigned int nvqs,
				struct virtqueue *vqs[],
				vq_callback_t *callbacks[],
				const char * const names[],
				const bool *ctx,
				struct irq_affinity *desc)
#else
static int sstar_rpmsg_find_vqs(struct virtio_device *vdev, unsigned int nvqs,
				struct virtqueue *vqs[],
				vq_callback_t *callbacks[],
				const char * const names[])
#endif
{
	struct sstar_virdev *virdev = to_sstar_virdev(vdev);
	struct sstar_rpmsg_vproc *rpdev = to_sstar_rpdev(virdev,
							virdev->base_vq_id / 2);
	int i, err;

	/* we maintain two virtqueues per remote processor (for RX and TX) */
	if (nvqs != 2)
		return -EINVAL;

	for (i = 0; i < nvqs; ++i) {
		vqs[i] = rp_find_vq(vdev, i, callbacks[i], names[i]);
		if (IS_ERR(vqs[i])) {
			err = PTR_ERR(vqs[i]);
			goto error;
		}
	}

	virdev->num_of_vqs = nvqs;

	virdev->nb.notifier_call = sstar_rpmsg_callback;
	sstar_rpmsg_register_nb((const char *)rpdev->rproc_name, &virdev->nb);

	return 0;

error:
	sstar_rpmsg_del_vqs(vdev);
	return err;
}

static void sstar_rpmsg_reset(struct virtio_device *vdev)
{
	dev_dbg(&vdev->dev, "reset !\n");
}

static u8 sstar_rpmsg_get_status(struct virtio_device *vdev)
{
	return 0;
}

static void sstar_rpmsg_set_status(struct virtio_device *vdev, u8 status)
{
	dev_dbg(&vdev->dev, "%s new status: %d\n", __func__, status);
}

static void sstar_rpmsg_vproc_release(struct device *dev)
{
	/* this handler is provided so driver core doesn't yell at us */
}

static struct virtio_config_ops sstar_rpmsg_config_ops = {
	.get_features	= sstar_rpmsg_get_features,
	.finalize_features = sstar_rpmsg_finalize_features,
	.find_vqs	= sstar_rpmsg_find_vqs,
	.del_vqs	= sstar_rpmsg_del_vqs,
	.reset		= sstar_rpmsg_reset,
	.set_status	= sstar_rpmsg_set_status,
	.get_status	= sstar_rpmsg_get_status,
};

static const struct of_device_id sstar_rpmsg_dt_ids[] = {
	{ .compatible = "sstar,sstar-rpmsg",},
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, sstar_rpmsg_dt_ids);

#define	SHARE_SIZE		(0x6000)
static u32 sstar_rpmsg_share_area;
static int set_vring_phy_buf(struct platform_device *pdev,
			struct sstar_rpmsg_vproc *rpdev, int vdev_nums)
{
	struct resource *res;
	resource_size_t size;
	unsigned int start, end;
	int i, ret = 0;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res && sstar_rpmsg_share_area != 0x0) {
		res = request_mem_region(sstar_rpmsg_share_area, SHARE_SIZE, "rpmsg");
	}

	if (res) {
		size = resource_size(res);
		start = res->start;
		end = res->start + size;
		for (i = 0; i < vdev_nums; i++) {
			rpdev->ivdev[i].vring[0] = start;
			rpdev->ivdev[i].vring[1] = start +
				RPMSG_RING_SIZE;
			start += RPMSG_RING_SIZE*2;
			if (start > end) {
				pr_err("Too small memory size %x!\n",
					(u32)size);
				ret = -EINVAL;
				break;
			}
		}
	} else {
		return -ENOMEM;
	}

	return ret;
}

static int rpmsg_irq_handler(void *arg)
{
	u32 idx;
	struct sstar_rpmsg_vproc *vproc = (struct sstar_rpmsg_vproc *)arg;

	DEFINE_WAIT(wait);
	while (1) {
		if (kthread_should_stop())
			break;

		prepare_to_wait(&vproc->ivdev[0].wq, &wait, TASK_UNINTERRUPTIBLE);
		idx = in_idx;
		if (idx == out_idx)
			schedule();
		finish_wait(&vproc->ivdev[0].wq, &wait);

		while (out_idx != idx) {
			blocking_notifier_call_chain(&(rpmsg_mbox.notifier), 0,
						NULL);
			out_idx++;
		}
	}
	return 0;
}

static int sstar_rpmsg_probe(struct platform_device *pdev)
{
	int i, j, ret = 0;
	struct device_node *np = pdev->dev.of_node;

	BLOCKING_INIT_NOTIFIER_HEAD(&(rpmsg_mbox.notifier));

	pr_info("RPMSG is ready for cross core communication!\n");

	for (i = 0; i < ARRAY_SIZE(sstar_rpmsg_vprocs); i++) {
		struct sstar_rpmsg_vproc *rpdev = &sstar_rpmsg_vprocs[i];

		ret = of_property_read_u32_index(np, "vdev-nums", i,
						&rpdev->vdev_nums);
		if (ret)
			rpdev->vdev_nums = 1;
		if (rpdev->vdev_nums > MAX_VDEV_NUMS) {
			pr_err("vdev-nums exceed the max %d\n", MAX_VDEV_NUMS);
			return -EINVAL;
		}

		if (!strcmp(rpdev->rproc_name, "rtos")) {
			ret = set_vring_phy_buf(pdev, rpdev,
						rpdev->vdev_nums);
			if (ret) {
				pr_err("No vring buffer.\n");
				return -ENOMEM;
			}
		} else {
			pr_err("No remote rtos processor.\n");
			return -ENODEV;
		}

		for (j = 0; j < rpdev->vdev_nums; j++) {
			pr_debug("%s rpdev%d vdev%d: vring0 0x%x, vring1 0x%x\n",
				__func__, i, rpdev->vdev_nums,
				rpdev->ivdev[j].vring[0],
				rpdev->ivdev[j].vring[1]);
			rpdev->ivdev[j].vdev.id.device = VIRTIO_ID_RPMSG;
			rpdev->ivdev[j].vdev.config = &sstar_rpmsg_config_ops;
			rpdev->ivdev[j].vdev.dev.parent = &pdev->dev;
			rpdev->ivdev[j].vdev.dev.release = sstar_rpmsg_vproc_release;
			rpdev->ivdev[j].base_vq_id = j * 2;
			init_waitqueue_head(&rpdev->ivdev[j].wq);
			rpdev->ivdev[j].task = kthread_create(rpmsg_irq_handler, rpdev,
							"rpmsg_irq/%d", j);
			if (IS_ERR(rpdev->ivdev[j].task)) {
				pr_err("%s failed to create irq worker for vdev %d:%d, err %ld",
					__func__, i, j, PTR_ERR(rpdev->ivdev[j].task));
				return PTR_ERR(rpdev->ivdev[j].task);
			}
			set_user_nice(rpdev->ivdev[j].task, MIN_NICE);
			wake_up_process(rpdev->ivdev[j].task);

			ret = register_virtio_device(&rpdev->ivdev[j].vdev);
			if (ret) {
				kthread_stop(rpdev->ivdev[j].task);
				pr_err("%s failed to register rpdev: %d\n",
					__func__, ret);
				return ret;
			}
		}
	}
	return ret;
}


static int disable_rtos;
module_param(disable_rtos, int, 0644);
MODULE_PARM_DESC(disable_rtos, "Disable RTOS RPMSG");

static struct platform_driver sstar_rpmsg_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name = "sstar-rpmsg",
		.of_match_table = sstar_rpmsg_dt_ids,
	},
	.probe = sstar_rpmsg_probe,
};

static int __init sstar_rpmsg_init(void)
{
	int ret;

	if (disable_rtos) {
		pr_info("RTOS RPMSG disabled - not registering driver");
		return 0;
	}

	ret = platform_driver_register(&sstar_rpmsg_driver);
	if (ret)
		pr_err("Unable to initialize rpmsg driver\n");
	else
		pr_info("sstar rpmsg driver is registered.\n");

	return ret;
}

/*
 * Use pure_initcall to make sure that it will be called on CPU0.
 */
int __init sstar_rpmsg_shm_init(void)
{
	if (disable_rtos) {
		pr_info("RTOS RPMSG disabled - not setting share area");
		return 0;
	}
#if defined(CONFIG_SS_AMP)
	interos_call_mbox_args_t *ptr_mbox_args;
	struct arm_smccc_res res;
	u32 marker;
	int count = 100;

	ptr_mbox_args = (interos_call_mbox_args_t *)(BASE_REG_MAILBOX_PA+BK_REG(0x60)+IO_OFFSET);
	ptr_mbox_args->arg0_l = 0;
	ptr_mbox_args->arg0_h = 0;
        ptr_mbox_args->ret_l = 0;
        ptr_mbox_args->ret_h = 0;

	arm_smccc_smc(INTEROS_SC_L2R_CALL, TARGET_BITS_CORE1, NSATT_BITS_GROUP0, SGIINTID_BITS_15,
		0, 0, 0, 0, &res);

	while (count > 0) {
		marker = (ptr_mbox_args->arg0_h << 16) + ptr_mbox_args->arg0_l;
		if (marker == 0xf1f1f1f1)
			break;
		udelay(1000);
		--count;
	}

	if (marker == 0xf3f2f1f0) {
		sstar_rpmsg_share_area = (u32)((ptr_mbox_args->ret_h << 16) + ptr_mbox_args->ret_l);
	} else {
		pr_err("Failed to get address of rpmsg share area\n");
		return 0;
	}
#else
	sstar_rpmsg_share_area = (u32)sstar_rpmsg_smc(INTEROS_SC_L2R_RPMSG_HANDSHAKE);
#endif
	printk(KERN_INFO "address of rpmsg share area:0x%x,%lu\n", sstar_rpmsg_share_area,
		RPMSG_RING_SIZE);
	return 0;
}
pure_initcall(sstar_rpmsg_shm_init);

MODULE_AUTHOR("SSTAR");
MODULE_DESCRIPTION("SSTAR remote processor messaging virtio device");
MODULE_LICENSE("GPL");
subsys_initcall(sstar_rpmsg_init);
